import streamlit as st
import requests
import openai

# Configuration
API_KEY = '33a3c8dfdd11322228be06d0994bf11a'
BASE_URL = 'https://api.themoviedb.org/3'
OPENAI_API_KEY = ''

# Set OpenAI API key
openai.api_key = OPENAI_API_KEY


# Helper functions
def fetch_data(endpoint, params):
    response = requests.get(f"{BASE_URL}/{endpoint}", params=params)
    return response.json()

def get_movie_details(movie_id):
    params = {'api_key': API_KEY, 'append_to_response': 'videos,credits'}
    return fetch_data(f"movie/{movie_id}", params)

def get_popular_movies(page=1):
    params = {'api_key': API_KEY, 'page': page}
    return fetch_data('movie/popular', params).get('results', [])

def search_movies(query, page=1):
    params = {'api_key': API_KEY, 'query': query, 'page': page, 'include_adult': False}
    response = fetch_data('search/movie', params)
    return response.get('results', [])

def get_movies_by_category(category_id, page=1):
    params = {'api_key': API_KEY, 'with_genres': category_id, 'page': page}
    return fetch_data('discover/movie', params).get('results', [])

def get_movie_genres():
    params = {'api_key': API_KEY}
    return fetch_data('genre/movie/list', params).get('genres', [])

def get_movie_recommendations(description):
    response = openai.ChatCompletion.create(
        model="gpt-3.5-turbo",
        messages=[
            {"role": "system", "content": "You are a movie recommendation assistant. First paragraph or your response is always a description of your prompt. Then list books in order, don't include year just title name."},
            {"role": "user", "content": f"Based on the following description, recommend a list of movies: {description}"}
        ],
        max_tokens=350
    )
    recommendations = response.choices[0].message['content'].strip()
    # Extract movie titles from the recommendation
    lines = recommendations.split('\n')
    if lines:
        return [line.split(' - ')[0].strip('"') for line in lines if line.strip()]
    return []

import re

def clean_title(title):
    match = re.search(r'\d*\.\s*["\'\s]*(.*?)["\'\s]*$', title)
    if match:
        return match.group(1).strip()
    return title.strip()

def sort_movies(movies, sort_by):
    if sort_by == "Rating (High to Low)":
        return sorted(movies, key=lambda x: x['vote_average'], reverse=True)
    elif sort_by == "Rating (Low to High)":
        return sorted(movies, key=lambda x: x['vote_average'])
    return movies

def display_movies(movies, is_recommendation_page=False):
    cols = st.columns(4)
    for idx, movie in enumerate(movies):
        with cols[idx % 4]:
            st.image(f"https://image.tmdb.org/t/p/w500{movie['poster_path']}", use_column_width=True)
            st.write(movie['title'])
            st.write(f"Rating: {movie['vote_average']}")
            if not is_recommendation_page:
                if st.button("More details", key=f"details_{movie['id']}_{idx}"):
                    st.experimental_set_query_params(page='movie', movie_id=movie['id'])
                    st.experimental_rerun()  # Force the page reload
            else:
                st.markdown(
                    f"""
                    <a href="/?page=movie&movie_id={movie['id']}" target="_blank">
                        <button style="
                            background-color: #202227; 
                            color: white; 
                            padding: 5px 10px; 
                            border: 1px solid grey; 
                            border-radius: 5px; 
                            margin-bottom: 1rem; 
                            font-size: 16px; 
                            cursor: pointer; 
                            transition: background-color 0.3s;
                        ">
                            More details
                        </button>
                    </a>
                    </a>
                    """,
                    unsafe_allow_html=True
                )


def display_movie_details(movie_id):
    movie_details = get_movie_details(movie_id)
    
    if st.button("Back"):
        # Clear URL parameters
        st.experimental_set_query_params()
        st.session_state['selected_movie_id'] = None
        st.session_state['page'] = 'Search/Home'
        st.experimental_rerun()
    
    st.markdown(
        f"""
        <div style='display: flex; align-items: flex-start; gap: 10px;'>
            <img src='https://image.tmdb.org/t/p/w500{movie_details["poster_path"]}' style='float: left; margin-right: 15px; width: 200px;'>
            <div>
                <h2>{movie_details['title']}</h2>
                <p>Rating: {movie_details['vote_average']}</p>
                <p>{movie_details['overview']}</p>
            </div>
        </div>
        """,
        unsafe_allow_html=True
    )
    
    genres = [genre['name'] for genre in movie_details.get('genres', [])]
    if genres:
        st.markdown(
            """
            <p style="margin-top:2rem">Genres</p>
            <div style='display: flex; flex-wrap: wrap; gap: 5px;'>
            {}
            </div>
            """.format(''.join(
                f"<span style='background-color: #202227; color: white; padding: 5px 10px; border-radius: 5px;'>{genre}</span>"
                for genre in genres
            )),
            unsafe_allow_html=True
        )
    
    cast = [member['name'] for member in movie_details.get('credits', {}).get('cast', [])[:5]]
    if cast:
        st.markdown(
            """
            <p style="margin-top:2rem">Actors</p>
            <div style='display: flex; flex-wrap: wrap; gap: 5px;margin-bottom:2rem'>
            {}
            </div>
            """.format(''.join(
                f"<span style='background-color: #343a40; color: white; padding: 5px 10px; border-radius: 5px;'>{actor}</span>"
                for actor in cast
            )),
            unsafe_allow_html=True
        )
    
    trailer = next((video['key'] for video in movie_details.get('videos', {}).get('results', []) if video['type'] == 'Trailer'), None)
    if trailer:
        st.subheader("Watch the trailer")
        st.video(f"https://www.youtube.com/watch?v={trailer}")

# Main application
st.markdown(
    "<div style='text-align: center;margin-bottom:3rem;'><img src='https://aleonel.com/assets/img/anthonyflix.png' width='300'><p>Welcome to my Netflix</p></div>",
    unsafe_allow_html=True
)

if 'page' not in st.session_state:
    st.session_state['page'] = 'Search/Home'
if 'selected_movie_id' not in st.session_state:
    st.session_state['selected_movie_id'] = None
if 'search_query' not in st.session_state:
    st.session_state['search_query'] = ""
if 'movies' not in st.session_state:
    st.session_state['movies'] = []
if 'page_number' not in st.session_state:
    st.session_state['page_number'] = 1
if 'is_loading' not in st.session_state:
    st.session_state['is_loading'] = False

# Get query parameters
query_params = st.experimental_get_query_params()
current_page = query_params.get("page", ["Search/Home"])[0]
movie_id = query_params.get("movie_id", [None])[0]



if current_page == "movie" and movie_id:
    display_movie_details(movie_id)
else:
    # Sidebar and main page logic
    page = st.sidebar.selectbox("Select a page", ["Search/Home", "Ai: Recommend Me a Movie"])

    if page == "Search/Home":
        search_query = st.text_input("", placeholder="Search for a movie...", value=st.session_state['search_query'], key="main_search")

        st.sidebar.header("Filter by Genre")
        genres = get_movie_genres()
        selected_genre_id = st.sidebar.selectbox("Genres", ["All"] + [genre['name'] for genre in genres])
        
        st.sidebar.header("Sort by")
        sort_by = st.sidebar.selectbox("Sort movies by", ["None", "Rating (High to Low)", "Rating (Low to High)"])

        if search_query != st.session_state['search_query']:
            st.session_state['search_query'] = search_query
            st.session_state['page_number'] = 1
            st.session_state['movies'] = []

        if search_query and selected_genre_id != 'All':
            st.write(f'Looking for "{search_query}" in "{selected_genre_id}"')
        elif search_query:
            st.write(f'Looking for "{search_query}"')
        
        if selected_genre_id and selected_genre_id != "All":
            genre_id = next((genre['id'] for genre in genres if genre['name'] == selected_genre_id), None)
            if genre_id:
                movies = get_movies_by_category(genre_id, st.session_state['page_number'])
        elif search_query:
            movies = search_movies(search_query, st.session_state['page_number'])
        else:
            movies = get_popular_movies(st.session_state['page_number'])
        
        movies = movies[:8]  # Load 8 movies at a time

        if st.session_state['is_loading']:
            st.session_state['movies'].extend(movies)
        else:
            st.session_state['movies'] = movies

        # Apply sorting
        st.session_state['movies'] = sort_movies(st.session_state['movies'], sort_by)
        
        display_movies(st.session_state['movies'])

        if st.button("Load More"):
            st.session_state['page_number'] += 1
            st.session_state['is_loading'] = True
            st.experimental_rerun()
        else:
            st.session_state['is_loading'] = False

    elif page == "Ai: Recommend Me a Movie":
        if st.session_state['selected_movie_id']:
            display_movie_details(st.session_state['selected_movie_id'])
        else:
            description = st.text_area("", height=75, placeholder="Describe something you want to watch today, or a feeling you want to feel")
            
            movies_not_found = 0
            if st.button("Get Recommendations"):
                if description:
                    with st.spinner("Fetching OpenAI to give recommendations..."):
                        recommendations = get_movie_recommendations(description)
                    
                    st.write(recommendations[0])
                    
                    recommended_movies = []
                    for title in recommendations[1:]:
                        cleaned_title = clean_title(title)
                        search_results = search_movies(cleaned_title)
                        
                        if search_results:
                            recommended_movies.append(search_results[0])
                        else:
                            st.write(f"{movies_not_found+1}. {cleaned_title}")
                            movies_not_found += 1
                    
                    # Limit to 20 recommended movies
                    recommended_movies = recommended_movies[:20]
                    
                    if recommended_movies:
                        display_movies(recommended_movies, is_recommendation_page=True)
                        if movies_not_found > 0:
                            st.markdown(
                                "<div style='background-color: rgba(0, 0, 0, 0.5); padding: 10px; border-radius: 5px;'><span style='background-color: rgba(0, 219, 211, 0.5); color: white; padding: 5px 10px; border-radius: 5px;margin-right:1rem;margin-bottom:1rem;'>Info</span>Some movies were found but we could not render preview on all of them due to API limitations, please look up remaining books online.</div>",
                                unsafe_allow_html=True
                            )
                    else:
                        st.markdown(
                            "<div style='background-color: rgba(0, 0, 0, 0.5); padding: 10px; border-radius: 5px;'><span style='background-color: rgba(255, 0, 0, 0.5); color: white; padding: 5px 10px; border-radius: 5px;margin-right:1rem;margin-bottom:1rem;'>Warning</span>Movies not found on API. Please search movies above manually if shown or try again.</div>",
                            unsafe_allow_html=True
                        )
                else:
                    st.error("Please enter a description.")
